<?php
declare(strict_types=1);

final class SettingsController {
  public static function publicSettings(): void {
    api_json(['ok'=>true, 'settings'=>public_settings()]);
  }

  public static function getAll(): void {
    require_role(['admin']);
    $rows = db()->query('SELECT key, value_json, updated_at FROM app_setting ORDER BY key')->fetchAll();
    $out = [];
    foreach ($rows as $r) {
      $out[] = ['key'=>$r['key'], 'value'=>json_decode($r['value_json'], true), 'updated_at'=>$r['updated_at']];
    }
    api_json(['ok'=>true, 'settings'=>$out]);
  }

  public static function update(): void {
    require_role(['admin']);
    $in = json_input();
    $key = (string)($in['key'] ?? '');
    $value = $in['value'] ?? null;
    if ($key === '' || !is_array($value)) api_error(400, 'Expected {key, value(object)}');
    settings_update($key, $value);
    // Emit event for live brand refresh
    EventsController::emitGlobal('settings.updated', ['key'=>$key]);
    api_json(['ok'=>true]);
  }

  public static function uploadAsset(): void {
    require_role(['admin']);
    if (empty($_FILES['file'])) api_error(400, 'Missing file');
    $key = (string)($_POST['key'] ?? ''); // branding.logoPath or branding.faviconPath
    if ($key === '') api_error(400, 'Missing key');

    $f = $_FILES['file'];
    if (($f['error'] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) api_error(400, 'Upload error');
    $name = $f['name'] ?? 'upload';
    $tmp = $f['tmp_name'] ?? '';
    $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
    $allowed = ['png','jpg','jpeg','svg','ico','webp'];
    if (!in_array($ext, $allowed, true)) api_error(400, 'Unsupported file type');

    $destName = bin2hex(random_bytes(16)) . '.' . $ext;
    $destRel = '/uploads/' . $destName;
    $destAbs = __DIR__ . '/../../public' . $destRel;

    if (!move_uploaded_file($tmp, $destAbs)) api_error(500, 'Could not store upload');

    $brand = setting('branding', []);
    if ($key === 'branding.logoPath') $brand['logoPath'] = $destRel;
    if ($key === 'branding.faviconPath') $brand['faviconPath'] = $destRel;
    settings_update('branding', $brand);

    EventsController::emitGlobal('branding.updated', ['branding'=>$brand]);
    api_json(['ok'=>true, 'path'=>$destRel]);
  }
}
